import Foundation

/// Life expectancy lookup tables based on WHO 2020 data for US/UK demographics
/// Provides baseline life expectancy calculations with age and gender stratification
struct LifeExpectancyTables {
    
    // MARK: - Types
    
    
    enum Country: String, CaseIterable {
        case us = "US"
        case uk = "UK"
    }
    
    struct LifeExpectancyData {
        let age: Int
        let lifeExpectancy: Double
    }
    
    // MARK: - WHO 2020 Life Table Data
    
    /// US Male life expectancy data (WHO 2020)
    private static let usMaleTable: [LifeExpectancyData] = [
        LifeExpectancyData(age: 0, lifeExpectancy: 76.3),
        LifeExpectancyData(age: 1, lifeExpectancy: 75.8),
        LifeExpectancyData(age: 5, lifeExpectancy: 71.9),
        LifeExpectancyData(age: 10, lifeExpectancy: 67.0),
        LifeExpectancyData(age: 15, lifeExpectancy: 62.0),
        LifeExpectancyData(age: 20, lifeExpectancy: 57.2),
        LifeExpectancyData(age: 25, lifeExpectancy: 52.4),
        LifeExpectancyData(age: 30, lifeExpectancy: 47.7),
        LifeExpectancyData(age: 35, lifeExpectancy: 43.0),
        LifeExpectancyData(age: 40, lifeExpectancy: 38.4),
        LifeExpectancyData(age: 45, lifeExpectancy: 33.9),
        LifeExpectancyData(age: 50, lifeExpectancy: 29.6),
        LifeExpectancyData(age: 55, lifeExpectancy: 25.5),
        LifeExpectancyData(age: 60, lifeExpectancy: 21.6),
        LifeExpectancyData(age: 65, lifeExpectancy: 18.1),
        LifeExpectancyData(age: 70, lifeExpectancy: 14.8),
        LifeExpectancyData(age: 75, lifeExpectancy: 11.9),
        LifeExpectancyData(age: 80, lifeExpectancy: 9.3),
        LifeExpectancyData(age: 85, lifeExpectancy: 7.1),
        LifeExpectancyData(age: 90, lifeExpectancy: 5.2),
        LifeExpectancyData(age: 95, lifeExpectancy: 3.8),
        LifeExpectancyData(age: 100, lifeExpectancy: 2.8)
    ]
    
    /// US Female life expectancy data (WHO 2020)
    private static let usFemaleTable: [LifeExpectancyData] = [
        LifeExpectancyData(age: 0, lifeExpectancy: 81.1),
        LifeExpectancyData(age: 1, lifeExpectancy: 80.5),
        LifeExpectancyData(age: 5, lifeExpectancy: 76.6),
        LifeExpectancyData(age: 10, lifeExpectancy: 71.6),
        LifeExpectancyData(age: 15, lifeExpectancy: 66.7),
        LifeExpectancyData(age: 20, lifeExpectancy: 61.8),
        LifeExpectancyData(age: 25, lifeExpectancy: 56.9),
        LifeExpectancyData(age: 30, lifeExpectancy: 52.0),
        LifeExpectancyData(age: 35, lifeExpectancy: 47.2),
        LifeExpectancyData(age: 40, lifeExpectancy: 42.4),
        LifeExpectancyData(age: 45, lifeExpectancy: 37.7),
        LifeExpectancyData(age: 50, lifeExpectancy: 33.0),
        LifeExpectancyData(age: 55, lifeExpectancy: 28.5),
        LifeExpectancyData(age: 60, lifeExpectancy: 24.2),
        LifeExpectancyData(age: 65, lifeExpectancy: 20.0),
        LifeExpectancyData(age: 70, lifeExpectancy: 16.1),
        LifeExpectancyData(age: 75, lifeExpectancy: 12.5),
        LifeExpectancyData(age: 80, lifeExpectancy: 9.4),
        LifeExpectancyData(age: 85, lifeExpectancy: 6.9),
        LifeExpectancyData(age: 90, lifeExpectancy: 4.9),
        LifeExpectancyData(age: 95, lifeExpectancy: 3.5),
        LifeExpectancyData(age: 100, lifeExpectancy: 2.5)
    ]
    
    /// UK Male life expectancy data (WHO 2020)
    private static let ukMaleTable: [LifeExpectancyData] = [
        LifeExpectancyData(age: 0, lifeExpectancy: 79.4),
        LifeExpectancyData(age: 1, lifeExpectancy: 78.8),
        LifeExpectancyData(age: 5, lifeExpectancy: 74.9),
        LifeExpectancyData(age: 10, lifeExpectancy: 69.9),
        LifeExpectancyData(age: 15, lifeExpectancy: 64.9),
        LifeExpectancyData(age: 20, lifeExpectancy: 60.0),
        LifeExpectancyData(age: 25, lifeExpectancy: 55.2),
        LifeExpectancyData(age: 30, lifeExpectancy: 50.3),
        LifeExpectancyData(age: 35, lifeExpectancy: 45.5),
        LifeExpectancyData(age: 40, lifeExpectancy: 40.7),
        LifeExpectancyData(age: 45, lifeExpectancy: 36.0),
        LifeExpectancyData(age: 50, lifeExpectancy: 31.4),
        LifeExpectancyData(age: 55, lifeExpectancy: 27.0),
        LifeExpectancyData(age: 60, lifeExpectancy: 22.8),
        LifeExpectancyData(age: 65, lifeExpectancy: 18.8),
        LifeExpectancyData(age: 70, lifeExpectancy: 15.1),
        LifeExpectancyData(age: 75, lifeExpectancy: 11.8),
        LifeExpectancyData(age: 80, lifeExpectancy: 8.9),
        LifeExpectancyData(age: 85, lifeExpectancy: 6.5),
        LifeExpectancyData(age: 90, lifeExpectancy: 4.6),
        LifeExpectancyData(age: 95, lifeExpectancy: 3.2),
        LifeExpectancyData(age: 100, lifeExpectancy: 2.3)
    ]
    
    /// UK Female life expectancy data (WHO 2020)
    private static let ukFemaleTable: [LifeExpectancyData] = [
        LifeExpectancyData(age: 0, lifeExpectancy: 82.9),
        LifeExpectancyData(age: 1, lifeExpectancy: 82.3),
        LifeExpectancyData(age: 5, lifeExpectancy: 78.4),
        LifeExpectancyData(age: 10, lifeExpectancy: 73.4),
        LifeExpectancyData(age: 15, lifeExpectancy: 68.4),
        LifeExpectancyData(age: 20, lifeExpectancy: 63.5),
        LifeExpectancyData(age: 25, lifeExpectancy: 58.5),
        LifeExpectancyData(age: 30, lifeExpectancy: 53.6),
        LifeExpectancyData(age: 35, lifeExpectancy: 48.7),
        LifeExpectancyData(age: 40, lifeExpectancy: 43.8),
        LifeExpectancyData(age: 45, lifeExpectancy: 39.0),
        LifeExpectancyData(age: 50, lifeExpectancy: 34.2),
        LifeExpectancyData(age: 55, lifeExpectancy: 29.6),
        LifeExpectancyData(age: 60, lifeExpectancy: 25.1),
        LifeExpectancyData(age: 65, lifeExpectancy: 20.8),
        LifeExpectancyData(age: 70, lifeExpectancy: 16.7),
        LifeExpectancyData(age: 75, lifeExpectancy: 12.9),
        LifeExpectancyData(age: 80, lifeExpectancy: 9.6),
        LifeExpectancyData(age: 85, lifeExpectancy: 6.8),
        LifeExpectancyData(age: 90, lifeExpectancy: 4.6),
        LifeExpectancyData(age: 95, lifeExpectancy: 3.1),
        LifeExpectancyData(age: 100, lifeExpectancy: 2.1)
    ]
    
    // MARK: - Public Methods
    
    /// Get baseline life expectancy for given demographics
    /// - Parameters:
    ///   - age: Current age in years (0-120)
    ///   - gender: Gender for demographic lookup
    ///   - country: Country for demographic lookup
    /// - Returns: Life expectancy in years, or nil if invalid parameters
    static func getLifeExpectancy(age: Double, gender: Gender, country: Country = .us) -> Double? {
        guard age >= 0 && age <= 120 else { return nil }
        
        let table = getTable(for: gender, country: country)
        return interpolateLifeExpectancy(age: age, table: table)
    }
    
    /// Get life expectancy with linear interpolation for precise age calculations
    /// - Parameters:
    ///   - age: Precise age (can include fractional years)
    ///   - table: Life expectancy table to use for lookup
    /// - Returns: Interpolated life expectancy
    private static func interpolateLifeExpectancy(age: Double, table: [LifeExpectancyData]) -> Double {
        let ageInt = Int(age)
        
        // Handle edge cases
        if age <= 0 {
            return table.first?.lifeExpectancy ?? 75.0 // Fallback value
        }
        
        if ageInt >= 100 {
            return table.last?.lifeExpectancy ?? 2.5 // Fallback for very old ages
        }
        
        // Find the two closest age points for interpolation
        var lowerBound: LifeExpectancyData?
        var upperBound: LifeExpectancyData?
        
        for data in table {
            if data.age <= ageInt {
                lowerBound = data
            } else {
                upperBound = data
                break
            }
        }
        
        // If exact match found
        if let lower = lowerBound, lower.age == ageInt {
            return lower.lifeExpectancy
        }
        
        // Perform linear interpolation
        guard let lower = lowerBound, let upper = upperBound else {
            // Fallback to nearest value
            return lowerBound?.lifeExpectancy ?? upperBound?.lifeExpectancy ?? 75.0
        }
        
        let ageDifference = Double(upper.age - lower.age)
        let lifeExpectancyDifference = upper.lifeExpectancy - lower.lifeExpectancy
        let ageOffset = age - Double(lower.age)
        
        return lower.lifeExpectancy + (lifeExpectancyDifference * ageOffset / ageDifference)
    }
    
    /// Get appropriate life expectancy table for demographics
    /// - Parameters:
    ///   - gender: Gender for table selection
    ///   - country: Country for table selection
    /// - Returns: Appropriate life expectancy table
    private static func getTable(for gender: Gender, country: Country) -> [LifeExpectancyData] {
        switch (country, gender) {
        case (.us, .male):
            return usMaleTable
        case (.us, .female):
            return usFemaleTable
        case (.us, .other):
            return usMaleTable // Default to male table for other gender
        case (.uk, .male):
            return ukMaleTable
        case (.uk, .female):
            return ukFemaleTable
        case (.uk, .other):
            return ukMaleTable // Default to male table for other gender
        }
    }
    
    /// Validate input parameters and return sanitized values
    /// - Parameters:
    ///   - age: Age to validate
    ///   - gender: Gender to validate
    ///   - country: Country to validate
    /// - Returns: Tuple of validated parameters or nil if invalid
    static func validateInputs(age: Double, gender: String, country: String) -> (age: Double, gender: Gender, country: Country)? {
        // Validate age bounds
        guard age >= 0 && age <= 120 else { return nil }
        
        // Validate gender
        guard let validGender = Gender(rawValue: gender.lowercased()) else { return nil }
        
        // Validate country (default to US if invalid)
        let validCountry = Country(rawValue: country.uppercased()) ?? .us
        
        return (age: age, gender: validGender, country: validCountry)
    }
    
    /// Get life expectancy adjustment for specific age ranges
    /// Used for more accurate calculations in different life phases
    /// - Parameters:
    ///   - age: Current age
    ///   - gender: Gender for demographic lookup
    ///   - country: Country for demographic lookup
    /// - Returns: Age-specific adjustment factor (typically 0.95-1.05)
    static func getAgeSpecificAdjustment(age: Double, gender: Gender, country: Country = .us) -> Double {
        switch age {
        case 0..<1:
            return 0.98 // Infant mortality considerations
        case 1..<15:
            return 1.02 // Lower mortality in childhood
        case 15..<25:
            return gender == .male ? 0.96 : 1.01 // Higher male risk in teens/early twenties
        case 25..<45:
            return 1.0 // Standard adult mortality
        case 45..<65:
            return 0.98 // Middle age health complications
        case 65..<80:
            return 0.97 // Increased elderly health risks
        default:
            return 0.95 // Advanced age complications
        }
    }
    
    /// Get life expectancy data for all ages in a demographic group
    /// Useful for generating mortality curves and validation
    /// - Parameters:
    ///   - gender: Gender for table selection
    ///   - country: Country for table selection
    /// - Returns: Complete life expectancy table
    static func getFullTable(for gender: Gender, country: Country = .us) -> [LifeExpectancyData] {
        return getTable(for: gender, country: country)
    }
}

// MARK: - Extensions

extension LifeExpectancyTables {
    /// Calculate remaining life expectancy with confidence intervals
    /// - Parameters:
    ///   - age: Current age
    ///   - gender: Gender for demographic lookup
    ///   - country: Country for demographic lookup
    /// - Returns: Tuple of (expected, lower bound, upper bound) life expectancy
    static func getLifeExpectancyWithConfidenceInterval(age: Double, gender: Gender, country: Country = .us) -> (expected: Double, lower: Double, upper: Double)? {
        guard let baseExpectancy = getLifeExpectancy(age: age, gender: gender, country: country) else {
            return nil
        }
        
        // Calculate 95% confidence interval (approximately ±2 years for most ages)
        let confidenceInterval = min(2.0, baseExpectancy * 0.1) // Max 2 years or 10% of expectancy
        
        return (
            expected: baseExpectancy,
            lower: max(0, baseExpectancy - confidenceInterval),
            upper: baseExpectancy + confidenceInterval
        )
    }
}